(function ($) {
    "use strict";
    if (typeof $ === "undefined") {
        console.error("jQuery is not loaded.");
        return;
    }
 const { __ } = wp.i18n;


    class PhoneVerification {
        constructor(form) {
            this.$form = $(form);
            this.formId = this.$form.data("form-id");
            if (!this.formId) return;

            this.settings = {};
            this.verified = false;
            this.otpLength = 6; // Default OTP length
            this.otpAttempts = 0;
            this.maxOtpAttempts = 5; // Maximum OTP verification attempts

            this.init();
        }

        async init() {
            try {
                const response = await this.ajax("settings");
                this.settings = response?.otp_settings ?? {};
                this.otpLength = parseInt(this.settings.count, 10) || 6;

                if (!this.isEnabled()) return;

                this.phoneField = this.$form.find(`#${this.settings.to}`);
                if (!this.phoneField.length) return;

                this.renderModal();
                this.bindFormSubmit();
            } catch (err) {
                console.error("Failed to load settings:", err);
            }
        }

        isEnabled() {
            return this.settings?.enable && this.settings.to !== "";
        }

        /** ---------- Modal ---------- */
        renderModal() {
            // Generate OTP input fields dynamically
            let otpInputs = '';
            for (let i = 0; i < this.otpLength; i++) {
                otpInputs += `<input type="text" class="pv-otp-input" maxlength="1" autocomplete="off" id="pv-otp-${i}" aria-label="OTP digit ${i + 1}" />`;
            }

            const modalHtml = `
                <div id="pv-modal" class="pv-modal" role="dialog" aria-labelledby="pv-title" aria-modal="true">
                    <div class="pv-modal-content">
                        <button id="pv-close" class="pv-close" aria-label="${__('Close', 'payamito-jet-engine')}">&times;</button>
                        <h2 id="pv-title">${__('Phone Verification', 'payamito-jet-engine')}</h2>
                        <p class="pv-phone"> <span id="pv-phone-display"></span></p>
                        <div class="pv-spinner"></div>

                        <div class="pv-step pv-step-request">
                            <button id="pv-request-btn" class="pv-btn pv-btn-primary"><span>${__('Request OTP', 'payamito-jet-engine')}</span><span class="spinner"></span></button>
                        </div>

                        <div class="pv-step pv-step-otp" style="display:none;">
                            <label for="pv-otp-0">${__('Verification Code', 'payamito-jet-engine')}</label>
                            <div class="pv-otp-container">${otpInputs}</div>
                            <button id="pv-verify-btn" class="pv-btn pv-btn-success"><span>${__('Verify', 'payamito-jet-engine')}</span><span class="spinner"></span></button>
                            <button id="pv-resend-btn" class="pv-btn pv-btn-secondary"><span>${PJE_FRONT_FORM.text.Resend}</span><span class="spinner"></span></button>
                            <div id="pv-timer" class="pv-timer"></div>
                        </div>

                        <div id="pv-message" class="pv-message"></div>
                    </div>
                </div>
            `;
            $("body").append(modalHtml);

            this.$modal      = $("#pv-modal");
            this.$spinner    = this.$modal.find(".pv-spinner");
            this.$requestBtn = this.$modal.find("#pv-request-btn");
            this.$verifyBtn  = this.$modal.find("#pv-verify-btn");
            this.$resendBtn  = this.$modal.find("#pv-resend-btn");
            this.$otpInputs  = this.$modal.find(".pv-otp-input");
            this.$message    = this.$modal.find("#pv-message");
            this.$timer      = this.$modal.find("#pv-timer");

            this.$requestBtn.on("click", () => this.requestOtp());
            this.$verifyBtn.on("click", () => this.verifyOtp());
            this.$resendBtn.on("click", () => this.resendOtp());
            this.$modal.find("#pv-close").on("click", () => this.closeModal());

            // Add event listeners for OTP inputs
            this.bindOtpInputs();
        }

        bindOtpInputs() {
            this.$otpInputs.each((index, input) => {
                $(input).on("input", (e) => {
                    const value = e.target.value;
                    // Allow only digits
                    if (value && !/^\d$/.test(value)) {
                        e.target.value = "";
                        return;
                    }
                    // Move to next input if a digit is entered
                    if (value.length === 1 && index < this.otpLength - 1) {
                        this.$otpInputs.eq(index + 1).focus();
                    }
                    // Auto-submit if all inputs are filled
                    const otp = this.$otpInputs
                        .map((_, inp) => $(inp).val())
                        .get()
                        .join("");
                    if (otp.length === this.otpLength) {
                        this.verifyOtp();
                    }
                });

                $(input).on("keydown", (e) => {
                    if (e.key === "Backspace" && !e.target.value && index > 0) {
                        // Move to previous input on backspace
                        this.$otpInputs.eq(index - 1).focus();
                    }
                });

                // Handle paste event
                $(input).on("paste", (e) => {
                    e.preventDefault();
                    const pastedData = (e.originalEvent.clipboardData || window.clipboardData).getData("text").replace(/\D/g, "");
                    if (pastedData.length) {
                        this.$otpInputs.each((i, inp) => {
                            if (i < pastedData.length && i < this.otpLength) {
                                $(inp).val(pastedData[i]);
                                if (i < this.otpLength - 1) {
                                    this.$otpInputs.eq(i + 1).focus();
                                }
                            }
                        });
                        // Auto-submit if all inputs are filled
                        const otp = this.$otpInputs
                            .map((_, inp) => $(inp).val())
                            .get()
                            .join("");
                        if (otp.length === this.otpLength) {
                            this.verifyOtp();
                        }
                    }
                });
            });
        }

        openModal() {
            const phone = this.phoneField.val().trim();
            if (!phone || !/^\+?\d{10,15}$/.test(phone)) {
                this.showMessage(__("Please enter a valid phone number", "payamito-jet-engine"), false);
                return;
            }
            $("#pv-phone-display").text(phone);
            this.$modal.addClass("active").fadeIn(300);
            // Focus on the request button for accessibility
            this.$requestBtn.focus();
        }

        closeModal() {
            this.$modal.fadeOut(300, () => {
                this.$modal.removeClass("active");
                this.$message.empty();
                this.$otpInputs.val("").removeClass("error");
                this.$timer.empty();
                this.$spinner.removeClass("active");
                this.$requestBtn.removeClass("loading").prop("disabled", false);
                this.$verifyBtn.removeClass("loading").prop("disabled", false);
                this.$resendBtn.removeClass("loading").prop("disabled", false);
                $(".pv-step-request").show();
                $(".pv-step-otp").hide();
                this.verified = false; // Reset verification state
                this.otpAttempts = 0; // Reset attempts
                // Remove hidden input to ensure OTP verification is required again
                this.$form.find('input[name="payamito_otp_verified"]').remove();
                // Re-enable form inputs and submit button
                this.$form.find(':input').prop('disabled', false);
                this.$form.find('button[type="submit"]').prop('disabled', false);
                this.$form.off("submit");
                this.bindFormSubmit();
            });
        }

        /** ---------- Form ---------- */
        bindFormSubmit() {
            this.$form.on("submit", (e) => {
                if (!this.verified) {
                    e.preventDefault();
                    this.openModal();
                }
            });
        }

        /** ---------- OTP Flow ---------- */
        async requestOtp() {
            try {
                this.$spinner.addClass("active");
                this.$requestBtn.addClass("loading").prop("disabled", true);
                const phone = this.phoneField.val().trim();
                const res = await this.ajax("request_otp", { phone });
                this.handleResponse(res, "request_otp");
            } catch (err) {
                this.showMessage(err.message || __("An error occurred. Please try again later.", "payamito-jet-engine"), false);
            } finally {
                this.$spinner.removeClass("active");
                this.$requestBtn.removeClass("loading").prop("disabled", false);
            }
        }

        async verifyOtp() {
            try {
                if (this.otpAttempts >= this.maxOtpAttempts) {
                    this.showMessage(__("You have reached the maximum number of verification attempts. Please try again later", "payamito-jet-engine"), false);
                    this.closeModal();
                    return;
                }
                const otp = this.$otpInputs
                    .map((_, input) => $(input).val())
                    .get()
                    .join("");
                if (otp.length !== this.otpLength) {
                    alert(__("Please enter a valid verification code", "payamito-jet-engine"));
                    return;
                }
                this.otpAttempts++;
                this.$spinner.addClass("active");
                this.$verifyBtn.addClass("loading").prop("disabled", true);
                this.$otpInputs.prop("disabled", true);
                const phone = this.phoneField.val().trim();
                const res = await this.ajax("verify_otp", { phone, otp });
                this.handleResponse(res, "verify_otp");
            } catch (err) {
                this.showMessage(err.message || __("An error occurred. Please try again later", "payamito-jet-engine"), false);
                this.$otpInputs.val("").removeClass("error");
                this.$otpInputs.first().focus();
            } finally {
                this.$spinner.removeClass("active");
                this.$verifyBtn.removeClass("loading").prop("disabled", false);
                this.$otpInputs.prop("disabled", false);
            }
        }

        async resendOtp() {
            try {
                this.$spinner.addClass("active");
                this.$resendBtn.addClass("loading").prop("disabled", true);
                this.$otpInputs.val("").removeClass("error");
                const phone = this.phoneField.val().trim();
                const res = await this.ajax("request_otp", { phone });
                this.handleResponse(res, "resend_otp");
            } catch (err) {
                this.showMessage(err.message || __("Failed to resend OTP", "payamito-jet-engine"), false);
            } finally {
                this.$spinner.removeClass("active");
                this.$resendBtn.removeClass("loading").prop("disabled", false);
            }
        }

        handleResponse(res, type) {
            if (!res) return;
            const success = !!res.result;
            this.showMessage(res.message || "", success);

            if (success) {
                if (type === "request_otp" || type === "resend_otp") {
                    $(".pv-step-request").hide();
                    $(".pv-step-otp").show();
                    this.$otpInputs.val("").removeClass("error");
                    this.$otpInputs.first().focus();
                    this.startTimer(); // Start timer after successful OTP request
                }
                if (type === "verify_otp") {
                    // Add hidden input as a flag
                    const hiddenInput = $('<input>').attr({
                        type: 'hidden',
                        name: 'payamito_otp_verified',
                        value: '1'
                    });
                    this.$form.append(hiddenInput);
                    this.verified = true;
                    this.closeModal();
                    this.$form.off("submit").submit();
                    // Remove hidden input and reset verification after submission
                    this.$form.on('submit', () => {
                        this.$form.find('input[name="payamito_otp_verified"]').remove();
                        this.verified = false; // Reset verification for next submission
                    });
                }
            } else if (type === "verify_otp") {
                // Highlight inputs as error and clear them
                this.$otpInputs.addClass("error");
                this.$otpInputs.val("");
                this.$otpInputs.first().focus();
            }
        }

        showMessage(msg, success) {
            this.$message.text(msg).css("color", success ? "#28a745" : "#dc3545").attr("role", "alert");
        }

        startTimer() {
            let seconds = parseInt(this.settings.resend || 30, 10);
            this.$resendBtn.prop("disabled", true);
            this.$timer.text(`${__("Resend available in", "payamito-jet-engine")} ${seconds} ${__("seconds", "payamito-jet-engine")}`);
            const interval = setInterval(() => {
                seconds--;
                this.$timer.text(`${__("Resend available in", "payamito-jet-engine")} ${seconds} ${__("seconds", "payamito-jet-engine")}`);
                if (seconds <= 0) {
                    clearInterval(interval);
                    this.$resendBtn.prop("disabled", false);
                    this.$timer.empty();
                }
            }, 1000);
        }

        ajax(actionType, dataSend = {}) {
            return new Promise((resolve, reject) => {
                $.ajax({
                    url: PJE_FRONT_FORM.ajaxurl,
                    type: "POST",
                    dataType: "json",
                    data: {
                        action: "payamito_jet_form_booking",
                        actionType,
                        formID: this.formId,
                        nonce: PJE_FRONT_FORM.nonce,
                        dataSend,
                    },
                })
                    .done((res) => {
                        if (res.success) {
                            resolve(res.data ?? {});
                        } else {
                            reject(res.data || { message: "Server error" });
                        }
                    })
                    .fail((xhr, status, error) => {
                        console.error(`AJAX error (${actionType}):`, status, error);
                        reject({ message: "Network error. Please try again." });
                    });
            });
        }
    }

    $(document).ready(() => {
        $(".jet-form, .jet-form-builder")
            .filter("form")
            .each((_, form) => new PhoneVerification(form));
    });
})(jQuery);